/**
 * uart.c UART Communication on the
 * STM32F103RB Nucleo Board
 *
 * @Author: Nicolas Dammin, 2016
 *
 */
#include "stm32f10x.h"
#include "uart.h"

void usart_init(uint32_t baudrate) {
	RCC_APB2PeriphClockCmd(USART_Port_Periph | RCC_APB2Periph_AFIO, ENABLE);

	if(USART_Periph == RCC_APB2Periph_USART1)
		RCC_APB2PeriphClockCmd(USART_Periph, ENABLE);
	else
		RCC_APB1PeriphClockCmd(USART_Periph, ENABLE);

	GPIO_InitTypeDef GPIO_Struct;
	USART_InitTypeDef USART_Struct;

	GPIO_Struct.GPIO_Pin = RX_Pin;
	GPIO_Struct.GPIO_Mode = GPIO_Mode_IN_FLOATING;
	GPIO_Init(USART_PORT, &GPIO_Struct);

	GPIO_Struct.GPIO_Pin = TX_Pin;
	GPIO_Struct.GPIO_Mode = GPIO_Mode_AF_PP;
	GPIO_Struct.GPIO_Speed = GPIO_Speed_10MHz;
	GPIO_Init(USART_PORT, &GPIO_Struct);

	USART_Struct.USART_BaudRate = baudrate;
	USART_Struct.USART_HardwareFlowControl = USART_HardwareFlowControl_None;
	USART_Struct.USART_Mode = USART_Mode_Rx | USART_Mode_Tx;
	USART_Struct.USART_Parity = USART_Parity_No;
	USART_Struct.USART_WordLength = USART_WordLength_8b;
	USART_Struct.USART_StopBits = USART_StopBits_1;
	USART_Init(USART_BASE, &USART_Struct);

	USART_Cmd(USART_BASE, ENABLE);

	//Send 1 dummybyte
	usart_sendChar('#');
}

void usart_sendChar(char c) {
	USART_SendData(USART_BASE, c);
	while(!USART_GetFlagStatus(USART_BASE, USART_FLAG_TC));
}

void usart_sendText(char *t) {
	while(*t != 0) {
		usart_sendChar(*t);
		t++;
	}
}

char usart_getChar() {
	char x;

	if(USART_GetFlagStatus(USART_BASE, USART_FLAG_RXNE)) {
		x = USART_ReceiveData(USART_BASE);
		USART_ClearFlag(USART_BASE, USART_FLAG_RXNE);
	} else {
		x = 0;
	}

	return x;
}

/**
 * Convert an 32-Bit integer to a string to output via Debug-USART
 */
void usart_sendNumber(uint32_t number)
{
	uint8_t i;
	char string[10];

	if(number == 0) {
		usart_sendChar('0');
		return;
	}

	for(i = 0; i < 10; i++)
	{
		if(number > 0)
		{
			string[9-i] = 48 + (number % 10);
			number /= 10;
		}
		else
		{
			string[9-i] = 0x00; //0
		}
	}

	for(i=0; i<10; i++)
	{
		if(string[i] != 0x00)
		{
			usart_sendChar(string[i]);
		}
	}
}
